/*

Copyright (c) 2004 PXI Project Team

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in the
Software without restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "PXIHyperTalkCompiler.h"
#include "variant.h"
#include <string>
#include <cstring>
#include <iostream>
#include <stack>
#include "PXIAbstractFramework.h"
using namespace std;

extern PXIAbstractFramework myAbstractFramework;

#define kIfSingle 0
#define kIfMulti 1

PXIHyperTalkCompiler::PXIHyperTalkCompiler( void ) : control_type_stack()
{
    source = "";
    write_symbols();
}

PXIHyperTalkCompiler::PXIHyperTalkCompiler( const string &s )
{
    source = s;
    write_symbols();
}

PXIHyperTalkCompiler::PXIHyperTalkCompiler( const PXIHyperTalkCompiler &c )
{
    source = c.source;
    write_symbols();
}

PXIHyperTalkCompiler::~PXIHyperTalkCompiler( void )
{
    /* Do nothing */;
}

void PXIHyperTalkCompiler::initialize( void )
{
	symbols.clear();
	source = "";
	composite_asmcode = "";
}

/**************************************************************
A note about the tokenizer:
	Our tokenizer was generously written by MrZ, and then
	mutilated beyond all recognition by creysoft. MrZ has
	made this code available under the MIT license.
***************************************************************/

const vector<Token> & PXIHyperTalkCompiler::tokenize( const string &inString )
{
    enum TokenState
	{
		TOKEN_STATE_WHITESPACE = 0,
		TOKEN_STATE_IDENTIFIER,
		TOKEN_STATE_STRING,
		TOKEN_STATE_OPERATOR
	}	state = TOKEN_STATE_WHITESPACE;

	int                 currOffset;
	const char          kQuote = 34;
	string				operatorChars( ",+-/<&>=^()\'[]" );
	string				whiteSpaceChars( " \t" );
	Token               currToken;
	vector<Token>		backupTokens;
	
	operatorChars += "*";

	currToken.mOffset = 0;

	for( currOffset = 0; currOffset < inString.length(); currOffset ++ )
	{
		char        currChar = inString.at( currOffset );
		int         opCharOffs = operatorChars.find( currChar );
	
		// Okay, so first we switch on the state, and
		// behave differently depending on what state
		// we're in. We start out as whitespace, because
		// we want to know when the first token starts.
		
		switch( state )
		{
		case TOKEN_STATE_WHITESPACE:
			if( whiteSpaceChars.find( currChar ) == -1 )    // No whitespace? We just started an identifier!
			{
				if( currChar == kQuote )
				{
					currToken.mType = state = TOKEN_STATE_STRING;
					currToken.mOffset = currOffset;
				}
				else if ( opCharOffs != -1 )
				{
					currToken.mType = state = TOKEN_STATE_OPERATOR;
					currToken.mString.append( 1, currChar );
					currToken.mOffset = currOffset;
				}
				else
				{
					currToken.mType = state = TOKEN_STATE_IDENTIFIER;   // Remember state change.
					currToken.mString.append( 1, currChar );            // Remember this char.
					currToken.mOffset = currOffset;                     // Remember token position.
				}
			}
			// else we don't care about whitespace and just forget it.
			break;
            
		case TOKEN_STATE_IDENTIFIER:
			if( whiteSpaceChars.find( currChar ) != -1 || currChar == kQuote || opCharOffs != -1 )    // Hit whitespace? Hit operator? End of token!
			{
				endToken( currToken );
				if( currChar == kQuote )
				{
					currToken.mType = state = TOKEN_STATE_STRING;
					currToken.mOffset = currOffset;
				}
				else if( opCharOffs != - 1 )
				{
					currToken.mType = state = TOKEN_STATE_OPERATOR;
					currToken.mString.append( 1, currChar );
					currToken.mOffset = currOffset;
				}
				else
					currToken.mType = state = TOKEN_STATE_WHITESPACE;
			}
			else
				currToken.mString.append( 1, currChar );    // Not whitespace? Add char to token!
			break;
			
		case TOKEN_STATE_STRING:
			if( currChar == kQuote )
			{
				currToken.mType = state = TOKEN_STATE_WHITESPACE;
				endToken( currToken ); // End the token
			}
			else
				currToken.mString.append( 1, currChar );    // Not endquote? Add char to token!
			break;
			
		case TOKEN_STATE_OPERATOR:
			if ( opCharOffs == -1 )
			{
				// It's not an operator. End the token.
				endToken( currToken );
				
				if ( currChar == kQuote )
				{
					currToken.mType = state = TOKEN_STATE_STRING;
					currToken.mOffset = currOffset;
				}
				else if ( whiteSpaceChars.find( currChar ) != -1 )
					state = TOKEN_STATE_WHITESPACE;
				else
				{
					currToken.mType = state = TOKEN_STATE_IDENTIFIER;
					currToken.mOffset = currOffset;
					currToken.mString.append( 1, currChar );
				}
			}
			else
								
				if ( currToken.mString == ">=" || currToken.mString == "<=" || currToken.mString == "&&" || currToken.mString == "<>" || currToken.mString == "--" )
				{
					endToken( currToken );
					currToken.mType = TOKEN_STATE_OPERATOR;
					currToken.mOffset = currOffset;					
					currToken.mString.append( 1, currChar );
				}
				else if
				(	    
						( ( currChar == '=' ) && inString.at( currOffset - 1 ) == '>' && inString.at( currOffset - 1 ) == '<' )
						|| ( ( currChar == '&' ) && inString.at( currOffset - 1 ) == '&' )
						|| ( ( currChar == '>' ) && inString.at( currOffset - 1 ) == '<' )
					    || ( ( currChar == '-' ) && inString.at( currOffset - 1 ) == '-' )
				)
				{
					currToken.mString.append( 1, currChar );
				}
				else
				{
					endToken( currToken );
					currToken.mType = TOKEN_STATE_OPERATOR;
					currToken.mOffset = currOffset;
					currToken.mString.append( 1, currChar );
				}
				
				    // Not whitespace? Add char to token!
			break;
		}
	}

	if( state == TOKEN_STATE_IDENTIFIER || state == TOKEN_STATE_STRING || state == TOKEN_STATE_OPERATOR )
		endToken( currToken );

	// Smart Re-Tokenizer

	// Copy symbols to backupTokens

	for( currOffset = 0; currOffset < symbols.size(); currOffset++ )
	{
		backupTokens.push_back( symbols[ currOffset ] );
	}

	symbols.clear();

	for( currOffset = 0; currOffset < backupTokens.size(); currOffset++ )
	{
		if ( ( ( backupTokens.size() - 1 - currOffset ) >= 4 ) && ! strcomp( backupTokens[ currOffset ].mString, "there" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 2 ].mString, "not " ) && ( ! strcomp( backupTokens[ currOffset + 3 ].mString, "a" ) || ! strcomp( backupTokens[ currOffset + 3 ].mString, "an" ) ) )
		{
			symbols.push_back( Token ( "there is not a", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 3;
		}
		else if ( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "there" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "is" ) && ( ! strcomp( backupTokens[ currOffset + 2 ].mString, "a" ) || ! strcomp( backupTokens[ currOffset + 2 ].mString, "an" ) ) )
		{
			symbols.push_back( Token ( "there is a", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if ( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "there" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 2 ].mString, "no" ) )
		{
			symbols.push_back( Token ( "there is no", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}		
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "does" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "not" ) && ! strcomp( backupTokens[ currOffset + 2].mString, "contain" ) )
		{
			symbols.push_back( Token ( "does not contain", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "not" ) && ! strcomp( backupTokens[ currOffset + 2].mString, "within" ) )
		{
			symbols.push_back( Token ( "is not within", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "not" ) && ! strcomp( backupTokens[ currOffset + 2].mString, "in" ) )
		{
			symbols.push_back( Token ( "is not in", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "not" ) && ( ! strcomp( backupTokens[ currOffset + 2 ].mString, "a" ) || ! strcomp( backupTokens[ currOffset + 2 ].mString, "an" ) ) )
		{
			symbols.push_back( Token ( "is not a", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 3 ) && ! strcomp( backupTokens[ currOffset ].mString, "isn" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "'" ) && ! strcomp( backupTokens[ currOffset + 2 ].mString, "t" ) )
		{
			symbols.push_back( Token ( "isn't", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 2;
		}
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 2 ) && ! strcomp( backupTokens[ currOffset ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "not" ) )
		{
			symbols.push_back( Token ( "is not", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 1;
		}		
		else if( ( ( backupTokens.size() - 1 - currOffset ) >= 2 ) && ! strcomp( backupTokens[ currOffset ].mString, "is" ) && ! strcomp( backupTokens[ currOffset + 1 ].mString, "in" ) )
		{
			symbols.push_back( Token( "is in", backupTokens[ currOffset ].mOffset, TOKEN_STATE_OPERATOR ) );
			currOffset += 1;
		}
		else
		{
			symbols.push_back( Token( backupTokens[ currOffset ].mString, backupTokens[ currOffset ].mOffset, backupTokens[ currOffset ].mType ) );
		}
	}

	return symbols;
}

void    PXIHyperTalkCompiler::endToken( Token& currToken )
{
    symbols.push_back( Token( currToken ) );
    currToken.mString.clear(); // Might be .erase()
}

PXIHyperTalkCompiler::searchRecord PXIHyperTalkCompiler::compile_container( const string & oper, int c )
{
	string s;
	string c_chunk, asmcode, extraStorage;
	stack < string > opstack;
	vector < string > s_tokens;
	int next_operator;
	int op_ptr ( 0 );
	bool theFlag (false);
	searchRecord r;
	
	asmcode = "";
	opstack.push( "@" );
	
	if ( ! strcomp( symbols[ c ].mString, "the" ) )
	{
		theFlag = true;
		c++;
	}
	
	if( ( is_a_chunk_word( symbols[ c ].mString ) ) && ( ! is_not_a_factor( symbols[ c + 1 ].mString ) ) )
	{
		s_tokens.push_back( "of" );
		s_tokens.push_back( "to" );
		s = "<make class chunk,class output>";
		do
		{
			c_chunk = symbols[ c ].mString;
			c ++;
			r = push_expr( c, "~", s_tokens );
			s += r.code;
			c = r.cur_ptr;
			op_ptr = r.n_optr;
			
			s += "<chunk_add ,class " + c_chunk + ">";
			
			if( r.found_token == "to" )
			{
				r = push_expr( c, "~", s_tokens );
				s += r.code;
				c = r.cur_ptr;
				op_ptr = r.n_optr;
				s += "<chunk_add ,class to>";
			}
			
		} while ( ( r.found_token == "of" ) && is_a_chunk_word( symbols[ c ].mString ) );
				
		r = push_factor( c, "~" );
		
		s += r.code;
		
		s += "<chunk_load ,class output>"; //<pop local " + symbols[ c ].mString + ",#nop>";
		
		r = compile_container( oper, c );
		
		asmcode = s + r.code;
		
		c = r.cur_ptr;
	}
	else if ( ! strcomp( symbols [ c + 1 ].mString, "of" ) )
	{
		asmcode = "<push const " + symbols[ c ].mString + ",#nop><swap ,>";
		
		r = push_factor ( c + 2, "~" );
		asmcode += r.code + "<swap ,><int ,const 7002>";
		
		r = compile_container( oper, c+2 );
		
		asmcode += r.code;
		
		c = r.cur_ptr;
	}
	else
	{
		
		// Logic:
		// Disable bracket processing, and do a speculative compile. If I have a trailing bracket,
		// then just write the appropriate logic, and return. Otherwise, I have to back up to where
		// I started from, and use the existing code. Good night.
				
		no_bracket_processing = true;
		
		r = push_factor ( c, "~" );

		no_bracket_processing = false;
		
		if ( symbols[ r.cur_ptr ].mString != "[" )
		{
			// No trailing bracket. Do it the easy way.
			
			if ( oper == "after" )
			{
				// This one's easy.
				asmcode = "<check_variable ,local " + symbols[ c ].mString + "><pop local " + symbols[ c ].mString + ",#cat>";
			}
			else if ( oper == "before" )
			{
				// This one's almost easy.
				asmcode = "<check_variable ,local " + symbols[ c ].mString + "><push local " + symbols[ c ].mString + ",#cat><pop local " + symbols[ c ].mString + ",#nop>";
			}
			else
			{
				//asmcode = "<check_variable ,local " + symbols[ c ].mString + "><pop local " + symbols[ c ].mString + ",#" + oper + ">";
				asmcode = makeSimpleContainerValue( "nop", symbols[ c ].mString );
			}
		}
		else
		{
			// Push the string onto the stack, and do a normal set.
			asmcode = r.code + "<swap ,>";
			s_tokens.clear();
			s_tokens.push_back( "]" );
			r = push_expr( r.cur_ptr + 1, "~", s_tokens );
			asmcode = r.code + "<swap ,>" + asmcode;
			asmcode += "<int ,const 7002>" + makeSimpleContainerValue( "nop", symbols[ c ].mString );
			c = r.cur_ptr;
		}
	}
	
	return searchRecord( asmcode, c, "" );
}

const string PXIHyperTalkCompiler::compile( const string &sr )
{
	int lineCounter;
	string scriptAssembler;
	cuteTokenizer lineSplitter;
	quotesrecord q;
	q.strip(sr);
	string s = replaceall( q.stripped_string, "\t", " " );
	fendif_counter = 0;
	
	lineSplitter.tokenize( s.substr( 0, ( int ) s.length() ), '\n' );
	
	last_line_was_jump = false;
	no_if_processing = false;
	control_type_stack.push( 0 );
		
	scriptAssembler = "<// __XION_STARTUP ><move const Done.,world .result><call ,label hstartup><jump ,label %xionexit>";
	
	for( lineCounter = 1; lineCounter < lineSplitter.numtokens(); lineCounter++ )
	{
		initialize();
		tokenize( lineSplitter.fetch_token( lineCounter ) );
		scriptAssembler += xcompile( 0, lineCounter );
	}
	
	scriptAssembler += "<// __XION_EXIT ><label ,label %xionexit><halt ,>";
	
	composite_asmcode = q.restore( scriptAssembler );
	return composite_asmcode;
}

const string &PXIHyperTalkCompiler::xcompile( int start_pos, int line_num )
{
	string next_operator, this_operator;
	string myasm;
    stack < string > opstack;
	vector < string > last_symbol;
    string lastOp, lastArg, curOp, curArg, sttValue, stpValue, thisLine, extraStorage;
    int c ( 0 );
	searchRecord r;
	
	int genericCounter, genericCounter2, genericCounter3, expCounter;
	
	symbols.push_back( Token( "@" ) );
	symbols.push_back( Token( "@" ) );
	
	if ( symbols[ start_pos ].mString == "--" || symbols[ start_pos ].mString == "@" )
	{
		return composite_asmcode = "";
	}
	
	myasm = "";
	thisLine = variant( line_num ).strVal();
	
	if( ( ! no_if_processing ) &&  control_type_stack.size() )
	{
		if ( ( last_line_was_jump && control_type_stack.top() == kIfSingle ) && strcomp( symbols[ start_pos ].mString, "else" ) )
		{
			myasm = "<label ,label %fendif" + master_control_stack.top().strVal() + ">";
			master_control_stack.pop();
			last_line_was_jump = false;
		}
	}
		
#pragma mark Compile : Put
	if ( ! strcomp( symbols[ start_pos ].mString, "put" ) )
	{
		last_symbol.push_back ( "@" );
		last_symbol.push_back ( "into" );
		last_symbol.push_back ( "before" );
		last_symbol.push_back ( "after" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
		
		if ( ! strcomp ( r.found_token, "into" ) )
		{
			r = compile_container ( "nop", r.cur_ptr );
			myasm += r.code;
		}
		else if ( ! strcomp( r.found_token, "before" ) )
		{
			r = compile_container ( "before", r.cur_ptr );
			myasm += r.code;
		}
		else if ( ! strcomp( r.found_token, "after" ) )
		{
			r = compile_container ( "after", r.cur_ptr );
			myasm += r.code;
		}
		else
		{
			myasm += "<int ,const 2>";
		}
	}
	
#pragma mark Compile : Choose
	else if ( ! strcomp( symbols[ start_pos ].mString, "choose" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = push_expr( start_pos + 2, "~", last_symbol );
		myasm += r.code;

		myasm += "<int ,const 6030>";
	}
	
#pragma mark Compile : Set
	else if ( ! strcomp( symbols[ start_pos ].mString, "set" ) )
	{
		if ( ! strcomp( symbols[ start_pos + 1 ].mString, "the" ) )
		{
			start_pos ++;
		}
		
		// Put the property name onto the stack.
		myasm += "<push const " + capalllower( symbols[ start_pos + 1 ].mString ) + ",#nop>";
		
		// Push the object onto the stack.
		last_symbol.push_back( "to" );
		genericCounter = start_pos;
		r = push_expr( start_pos + 3, "~", last_symbol );
		myasm += r.code;
		
		// Push the value onto the stack.
		last_symbol.pop_back();
		last_symbol.push_back( "@" );
		r = push_expr( r.cur_ptr, "~", last_symbol );
		myasm += r.code;
		
		// Eat the property name, object, and value.
		myasm += "<int ,const 7002>";
		
		// Save the current position, reset it, and
		// recompile the object location as an input
		// container. Then reset start_pos for whatever
		// nefarious purpose...
		genericCounter2 = start_pos;
		start_pos = genericCounter;
		r = compile_container( "nop", start_pos + 3 );
		start_pos = genericCounter2;
		myasm += r.code;
	}
	
#pragma mark Compile : Add
	else if ( ! strcomp( symbols[ start_pos ].mString, "add" ) )
	{
		last_symbol.push_back ( "to" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
		
		r = compile_container ( "add", r.cur_ptr );
		myasm += r.code;
	}
	
#pragma mark Compile : Subtract
	else if ( ! strcomp( symbols[ start_pos ].mString, "subtract" ) )
	{
		last_symbol.push_back ( "from" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
		
		r = compile_container ( "sub", r.cur_ptr );
		myasm += r.code;
	}
	
#pragma mark Compile : Multiply
	else if ( ! strcomp( symbols[ start_pos ].mString, "multiply" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = compile_container ( "mul", start_pos + 1 );
		myasm += r.code;
		
		r = push_expr( r.cur_ptr + 2, "~", last_symbol );
		myasm = r.code + myasm;
	}	
	
#pragma mark Compile : Divide
	else if ( ! strcomp( symbols[ start_pos ].mString, "divide" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = compile_container ( "div", start_pos + 1 );
		myasm += r.code;
		
		r = push_expr( r.cur_ptr + 2, "~", last_symbol );
		myasm = r.code + myasm;
	}	

#pragma mark Compile : Get
	else if ( ! strcomp( symbols[ start_pos ].mString, "get" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
	
		myasm += "<pop local it,#nop>";
	}
	
#pragma mark Compile : Ask
	else if ( ! strcomp( symbols[ start_pos ].mString, "ask" ) )
	{
		last_symbol.push_back ( "@" );
		last_symbol.push_back ( "with" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;

		if ( ! strcomp ( r.found_token, "with" ) )
		{
			last_symbol.clear();
			last_symbol.push_back ( "@" );

			r = push_expr ( r.cur_ptr, "~", last_symbol );
			myasm += r.code;
		}
		else
		{
			myasm += "<push const \\n,#nop>";
		}
		
		myasm += "<int ,const 6050><pop local it,#nop>";
		
	}
	
#pragma mark Compile : Answer
	else if ( ! strcomp( symbols[ start_pos ].mString, "answer" ) )
	{
		last_symbol.push_back ( "@" );
		last_symbol.push_back ( "with" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
		
		if ( ! strcomp ( r.found_token, "with" ) )
		{
			last_symbol.clear();
			last_symbol.push_back ( "@" );
			last_symbol.push_back ( "or" );
			
			r = push_expr ( r.cur_ptr, "~", last_symbol );
			myasm += r.code;
			
			if ( ! strcomp ( r.found_token, "or" ) )
			{			
				r = push_expr ( r.cur_ptr, "~", last_symbol );
				myasm += r.code;
				
				if ( ! strcomp ( r.found_token, "or" ) )
				{	
					last_symbol.clear();
					last_symbol.push_back ( "@" );
					
					r = push_expr ( r.cur_ptr, "~", last_symbol );
					myasm += r.code;
				}
				else
				{
					myasm += "<push const \\n,#nop>";
				}		
			}
			else
			{
				myasm += "<push const \\n,#nop><push const \\n,#nop>";
			}		
			
		}
		else
		{
			myasm += "<push const \\n,#nop><push const \\n,#nop><push const \\n,#nop>";
		}
		
		myasm += "<int ,const 6051><pop local it,#nop>";
		
	}
	
#pragma mark Compile : If
	else if( ! strcomp( symbols[ start_pos ].mString, "if" ) )
	{
		last_line_was_jump = true;
		
		master_control_stack.push( variant( line_num ) );

		last_symbol.push_back ( "then" );
		r = push_expr( start_pos + 1, "~", last_symbol );
		
		myasm += r.code;
		
		if( strcomp( symbols[ r.cur_ptr + 1 ].mString, "@" ) )
		{
			// Single line behavior.
			control_type_stack.push( kIfSingle );
			no_if_processing = true;
			myasm += "<pop local .condition,#nop><jump_if_mem_false local .condition,label %endif" + variant( line_num ).strVal() + ">" + xcompile( r.cur_ptr, line_num ) + "<jump ,label %fendif" + master_control_stack.top().strVal() + "><label ,label %endif" + variant( line_num ).strVal() + ">";
			no_if_processing = false;
		}
		else
		{
			// Multiline behavior.
			control_stack.push( variant( line_num ) );
			control_type_stack.push( kIfMulti );
			myasm += "<pop local .condition,#nop><jump_if_mem_false local .condition,label %endif" + control_stack.top().strVal() + ">";
		}
	}
#pragma mark Compile : Else If
	else if( ( ! strcomp( symbols[ start_pos ].mString, "else" ) ) && ( ! strcomp( symbols[ start_pos + 1 ].mString, "if" ) ) )
	{
		// else-if behavior.
		
		last_line_was_jump = true;
		
		if ( control_type_stack.size() )
		{
			if( control_type_stack.top() == kIfMulti )
			{
				no_if_processing = true;
				myasm = "<jump ,label %fendif" + master_control_stack.top().strVal() + "><label ,label %endif" + control_stack.top().strVal() + ">";
				no_if_processing = false;
			}
			control_type_stack.pop();
		}
		
		last_symbol.push_back ( "then" );
		r = push_expr( start_pos + 2, "~", last_symbol );
		
		myasm += r.code;
		
		if( strcomp( symbols[ r.cur_ptr + 1 ].mString, "@" ) )
		{
			control_type_stack.push( kIfSingle );
			no_if_processing = true;
			myasm += "<pop local .condition,#nop><jump_if_mem_false local .condition,label %endif" + variant( line_num ).strVal() + ">" + xcompile( r.cur_ptr, line_num ) + "<jump ,label %fendif" + master_control_stack.top().strVal() + "><label ,label %endif" + variant( line_num ).strVal() + ">";
			no_if_processing = false;
		}
		else
		{
			control_type_stack.push( kIfMulti );
			control_stack.push( variant( line_num ) );
			myasm += "<pop local .condition,#nop><jump_if_mem_false local .condition,label %endif" + control_stack.top().strVal() + ">";
		}
	}
	
#pragma mark Compile : Else
	else if( ! strcomp( symbols[ start_pos ].mString, "else" ) )
	{
		
		last_line_was_jump = true;
				
		if ( control_type_stack.size() )
		{
			if( control_type_stack.top() == kIfMulti )
			{
				myasm = "<jump ,label %fendif" + master_control_stack.top().strVal() + "><label ,label %endif" + control_stack.top().strVal() + ">";
				control_type_stack.pop();
			}
			control_type_stack.pop();
		}
		
		if ( strcomp( symbols[ start_pos + 1 ].mString, "@" ) )
		{
			// Single else behavior
			no_if_processing = true;
			myasm += xcompile( start_pos + 1, line_num ) + "<jump ,label %fendif" + master_control_stack.top().strVal() + "><label ,label %endif" + variant( line_num ).strVal() + ">" ;
			no_if_processing = false;
			control_type_stack.push( kIfSingle );
			control_stack.push( variant( line_num ) );
		}
		else
		{
			// Standard else behavior.
			control_stack.push( variant( line_num ) );
			control_type_stack.push( kIfMulti );
		}
	}
	
#pragma mark Compile : Repeat Until
	else if( ! strcomp( symbols[ start_pos ].mString, "repeat" ) && ! strcomp( symbols[ start_pos + 1 ].mString, "until" ) )
	{
		loop_stack.push( variant( line_num ) );
		
		last_symbol.push_back ( "@" );
		r = push_expr( start_pos + 2, "~", last_symbol );
				
		myasm += "<jump ,label %loopmain" + thisLine + "><label ,label %looptop" + thisLine + ">" + string( r.code ) + "<pop local %loopnanny" + thisLine + ",#nop><jump_if_mem_true local %loopnanny" + thisLine + ",label %loopend" + thisLine + "><label ,label %loopmain" + thisLine + ">";
	}
	
#pragma mark Compile : Repeat While
	else if( ! strcomp( symbols[ start_pos ].mString, "repeat" ) && ! strcomp( symbols[ start_pos ].mString, "while" ) )
	{
		loop_stack.push( variant( line_num ) );
		
		last_symbol.push_back ( "@" );
		r = push_expr( start_pos + 2, "~", last_symbol );
		
		myasm += "<jump ,label %loopmain" + thisLine + "><label ,label %looptop" + thisLine + ">" + string( r.code ) + "<pop local %loopnanny" + thisLine + ",#nop><jump_if_mem_false local %loopnanny" + thisLine + ",label %loopend" + thisLine + "><label ,label %loopmain" + thisLine + ">";
	}
	
#pragma mark Compile : Repeat With
	else if( ! strcomp( symbols[ start_pos ].mString, "repeat" ) && ! strcomp( symbols[ start_pos + 1 ].mString, "with" ) )
	{
		loop_stack.push( variant( line_num ) );
		last_symbol.push_back( "to" );
		last_symbol.push_back( "down" );
		last_symbol.push_back( "@" );
		
		r = push_expr( start_pos + 4, "~", last_symbol );
		sttValue = r.code;
		
		if ( ! strcomp ( r.found_token, "down" ) )
		{
			r = push_expr( start_pos + 7, "~", last_symbol );
		}
		else
		{
			r = push_expr( start_pos + 6, "~", last_symbol );
		}
		stpValue = r.code;
		
		myasm += sttValue + "<pop local %loopnannystart" + thisLine + ",#nop><subtract const 1,local %loopnannystart" + thisLine + ">";
		myasm += stpValue + "<pop local %loopnannyend" + thisLine + ",#nop><label ,label %looptop" + thisLine + "><push local %loopnannyend" + thisLine + ",#nop><push local %loopnannystart" + thisLine + ",#cmp><pop local .condition,#nop><jump_if_mem_true local .condition,label %loopend" + thisLine + "><add const 1,local %loopnannystart" + thisLine + "><move local %loopnannystart" + thisLine + ",local " + symbols[ start_pos + 2 ].mString + ">";
	}
	
#pragma mark Compile : Repeat
	else if( ! strcomp( symbols[ start_pos ].mString, "repeat" ) )
	{
		loop_stack.push( variant( line_num ) );
		
		last_symbol.push_back ( "times" );
		r = push_expr( start_pos + 1, "~", last_symbol );
		
		myasm += r.code + "<pop local %loopnanny" + thisLine + ",#nop><add const 1,local %loopnanny" + thisLine + "><label ,label %looptop" + thisLine + "><subtract const 1,local %loopnanny" + thisLine + "><push local %loopnanny" + thisLine + ",#nop><push const 0,#cmp><pop local .condition,#nop><jump_if_mem_true local .condition,label %loopend" + thisLine + ">";
	}
	
#pragma mark Compile : On
	else if ( ! strcomp( symbols[ 0 ].mString, "on" ) )
	{
		myasm = "<proc ,label h" + symbols[ 1 ].mString + "><pop local .returnaddr,#nop>";
		if ( symbols[ 2 ].mString != "@" )
		{
			genericCounter = 2;
			while( symbols[ genericCounter ].mString != "@" )
			{
				myasm += "<pop local " + symbols[ genericCounter++ ].mString + ",#nop>";
			}
		}
	}
#pragma mark Compile : On
	else if ( ! strcomp( symbols[ 0 ].mString, "on" ) )
	{
		myasm = "<proc ,label h" + symbols[ 1 ].mString + "><pop local .returnaddr,#nop>";
		if ( symbols[ 2 ].mString != "@" )
		{
			genericCounter = 2;
			while( symbols[ genericCounter ].mString != "@" )
			{
				myasm += "<pop local " + symbols[ genericCounter++ ].mString + ",#nop>";
			}
		}
	}
#pragma mark Compile : Global
	else if ( ! strcomp( symbols[ 0 ].mString, "global" ) )
	{		
		if ( symbols[ 1 ].mString != "@" )
		{
			genericCounter = 1;
			while( symbols[ genericCounter ].mString != "@" )
			{
				// Check to see if we've already declared this global or not.
				if( ! haveCurrentGlobal( capalllower( symbols[ genericCounter++ ].mString ) ) )
				{
					// If we haven't, go ahead and insert the location, with the name as a key.
					current_globals.insert( pair< string, int >
											( capalllower( symbols[ genericCounter - 1 ].mString ),
											myAbstractFramework.getGlobalVariableIndex( capalllower( symbols[ genericCounter - 1 ].mString ) ) ) );
				}
			}
		}
	}
#pragma mark Compile : End
	else if ( ! strcomp( symbols[ 0 ].mString, "end" ) )
	{

		// Clear the list of global variables so we can start over.
		current_globals.clear();

		if( ! strcomp( symbols[ 1 ].mString, "if" ) )
		{
			myasm = "<label ,label %endif" + control_stack.top().strVal() + "><label ,label %fendif" + master_control_stack.top().strVal() + ">";
			master_control_stack.pop();
			control_stack.pop();
			control_type_stack.pop();
			last_line_was_jump = false;
		}
		else if( ! strcomp( symbols[ 1 ].mString, "repeat" ) )
		{
			myasm += "<jump ,label %looptop" + loop_stack.top().strVal() + "><label ,label %loopend" + loop_stack.top().strVal() + ">";
			loop_stack.pop();
		}
		else
		{
			myasm += "<move const  ,world .result><jump_mem ,local .returnaddr><endp ,>";
		}
	}
	else if( ! strcomp( symbols[ start_pos ].mString, "return" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = push_expr( start_pos + 1, "~", last_symbol );
		myasm += r.code;
		
		myasm += "<pop world .result,#nop><jump_mem ,local .returnaddr>";
	}
#pragma mark Compile : Create Window
	else if ( ! strcomp( symbols[ start_pos ].mString, "create" ) && ! strcomp( symbols[ start_pos + 1 ].mString, "window" ) )
	{
		last_symbol.push_back ( "@" );
		
		r = push_expr( start_pos + 2, "~", last_symbol );
		myasm += r.code;
		
		myasm += "<int ,const 6000>";
	}
	else
	{
		// Must be a handler call.
		
		extraStorage = myasm;
		myasm = "";
		
		if ( symbols[ start_pos + 1 ].mString != "@" )
		{
			expCounter = 1;
		
			last_symbol.push_back( "@" );
			last_symbol.push_back( "," );
			
			r = push_expr( start_pos + expCounter, "~", last_symbol );
			
			myasm = r.code + myasm;
			expCounter = r.cur_ptr + 1;
			
			while( r.found_token == "," )
			{
				r = push_expr( start_pos + expCounter, "~", last_symbol );
				
				myasm = r.code + myasm;
				expCounter = r.cur_ptr + 1;
			}
		}
		
		myasm = "<call ,label h" + symbols[ start_pos ].mString + ">" + myasm;
		
		//tempcode += extraStorage + "<call ,label " + ( * line_map.find( "0" ) ).second + ">";
		//return tempcode;
	}
	
	return composite_asmcode = myasm;
}

PXIHyperTalkCompiler::searchRecord PXIHyperTalkCompiler::push_expr( int pos, string oper, const vector< string > & search_tokens )
{	
    stack < string > opstack;
    string lastOp, lastArg, curOp, curArg, asmcode;
    int c ( pos ), i ( 0 );
	string s, this_operator, next_operator;
	searchRecord r;
	
	opstack.push( "@" );
	
	r = push_factor( c, "~" );
	asmcode += r.code;
	c = r.cur_ptr + 1;
	this_operator = "~";
	next_operator = symbols[ c-1 ].mString;
	
	while( c < ( int )symbols.size() - 1 )
	{
		for( i = 0; i < ( int ) search_tokens.size(); ++i )
		{
			if( strcmpi( symbols[ c - 1 ].mString.c_str(), search_tokens[ i ].c_str() ) == 0 )
			{
				goto parse_expr_loopdone; // See: tokenize()
			}
		}
		
		r = push_factor( c, "~" );
		c = r.cur_ptr + 1;
		this_operator = next_operator;
		next_operator = symbols[ c-1 ].mString;
		
		while( compare_precedence( opstack.top(), this_operator ) >= 0 )
		{
			asmcode += "<merge ,#" + makeInlineOperator( opstack.top() ) + ">";
			opstack.pop();
		}

		asmcode += r.code;

		if( compare_precedence( this_operator, next_operator ) == -1 )
		{
			opstack.push( this_operator );
		}
		else
		{ 
			asmcode += "<merge ,#" + makeInlineOperator( this_operator ) + ">";
		}
	}
	parse_expr_loopdone:
		
	while( opstack.top() != "@" )
	{
		asmcode += "<merge ,#" + makeInlineOperator( opstack.top() ) + ">";
		opstack.pop();
	}
	
	if ( i > search_tokens.size() - 1 )
	{
		i = 0;
	}
	
	return searchRecord( asmcode, c, search_tokens[ i ] );
}

PXIHyperTalkCompiler::searchRecord PXIHyperTalkCompiler::push_factor( int pos, string oper )
{
	string s,sx,sy;
	int i;
	string backup_oper;
	stack< string > s2;
	bool no_optimize ( false );
	bool the_flag ( false );
	bool dontProcessTrailingBrackets( no_bracket_processing );
	pair < string, int > temp;
	vector < string > s_tokens;
	searchRecord r;
	int optr_inc = 0;
	
	no_bracket_processing = false;
	
	// Unary operators.
	while( ( symbols[ pos ].mString == "not" ) || ( symbols[ pos ].mString == "-" ) || ( symbols[ pos ].mString == "+" ) )
	{
		no_optimize = true;
		s2.push( symbols[ pos ].mString );
		pos ++;
	}
	
	if( no_optimize && oper != "~" )
	{
		backup_oper = oper;
		oper = "~";
	}
	else
	{
		no_optimize = false;
	}
	
	if( strcmpi( symbols[ pos ].mString.c_str(), "the" ) == 0 )
	{
		the_flag = true;
		pos++;
	}
	
	if( is_a_number( symbols[ pos ].mString ) || ( symbols[ pos ].mString[ 0 ] == 1 ) )
	{
		s += makeSimpleValue( oper, symbols[ pos ].mString );
	}
	else if ( symbols[ pos ].mString == "(" )
	{
		pos++;
		s_tokens.push_back( ")" );
		r = push_expr( pos, oper, s_tokens );
		s += r.code;
		pos = r.cur_ptr - 1;
	}
	else if ( ( strcomp( symbols[ pos ].mString, "middle" ) == 0 ) || ( strcomp( symbols[ pos ].mString, "mid" ) == 0 ) )
	{
		pos ++;
		if( is_a_chunk_word( symbols[ pos ].mString ) )
		{
			s = "<make class chunk,class input><push const 2147483647,#nop><chunk_add ,class " + symbols[ pos ].mString + ">";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr -1;
			s += "<chunk_load ,>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if ( ( strcomp( symbols[ pos ].mString, "middle" ) == 0 ) || ( strcomp( symbols[ pos ].mString, "mid" ) == 0 ) )
	{
		pos ++;
		if( is_a_chunk_word( symbols[ pos ].mString ) )
		{
			s = "<make class chunk,class input><push const 2147483647,#nop><chunk_add ,class " + symbols[ pos ].mString + ">";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr -1;
			s += "<chunk_load ,>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if ( strcomp( symbols[ pos ].mString, "mdd" ) == 0 )
	{
		pos ++;
		if( is_a_chunk_word( symbols[ pos ].mString ) )
		{
			s = "<make class chunk,class input><push const -2147483648,#nop><chunk_add ,class " + symbols[ pos ].mString + ">";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr -1;
			s += "<chunk_load ,>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if ( ( is_tens_place_ordinal( symbols[ pos ].mString ) ) && ( symbols[ pos + 1 ].mString == "-" ) && ( is_ones_place_ordinal( symbols[ pos + 2 ].mString ) ) )
	{
		pos ++;
		pos ++; // skip past the -
		pos ++; // skip past ones place ordinal
		if( is_a_chunk_word( symbols[ pos ].mString ) )
		{
			s = "<make class chunk,class input><push const " + variant( convert_ordinal( symbols[ pos - 3 ].mString + "_" + symbols[ pos - 1 ].mString ) ).strVal() + ",#nop><chunk_add ,class " + symbols[ pos ].mString + ">";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr -1;
			s += "<chunk_load ,>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}		
	}
	else if ( ( i = convert_ordinal( symbols[ pos ].mString ) ) != -2 )
	{
		pos ++;
		if( is_a_chunk_word( symbols[ pos ].mString ) )
		{
			s = "<make class chunk,class input><push const " + variant( i ).strVal() + ",#nop><chunk_add ,class " + symbols[ pos ].mString + ">";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr -1;
			s += "<chunk_load ,>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if ( strcomp( symbols[ pos ].mString, "number" ) == 0 )
	{
		pos ++;
		pos ++;
		if ( is_a_chunk_word( symbols[ pos ].mString.substr( 0, ( int ) symbols[ pos ].mString.size() - 1 ) ) )
		{
			s = "<push const " + symbols[ pos ].mString + ",#nop>";
			pos ++; // Skip past the 'of'
			pos ++;
			r = push_factor( pos, "~" );
			s += r.code;
			pos = r.cur_ptr - 1;
			s += "<int ,const 5001>";
		}
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if ( ( is_a_chunk_word( symbols[ pos ].mString ) ) && ( ! is_not_a_factor( symbols[ pos + 1 ].mString ) ) )
	{
		s2.push( symbols[ pos ].mString );
		pos++;
		s += "<make class chunk,class input>";
		s_tokens.push_back( "of" );
		s_tokens.push_back( "to" );
		r = push_expr( pos, "~", s_tokens );
		s += r.code;
		pos = r.cur_ptr - 1;
		s += "<chunk_add ,class " + s2.top() + ">";
		s2.pop();
		if( strcmpi( r.found_token.c_str(), "to" ) == 0 )
		{
			pos++;
			s_tokens.pop_back();
			r = push_expr( pos, "~", s_tokens );
			s += r.code;
			pos = r.cur_ptr - 1;
			s += "<chunk_add ,class to>";
		}
		pos++;
		r = push_factor( pos, "~" );
		s += r.code;
		pos = r.cur_ptr - 1;
		s += "<chunk_load ,>";
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
	}
	else if( i = is_a_shortfunc( symbols[ pos ].mString ) )
	{
		pos++;
		
		if( strcmpi( symbols[ pos ].mString.c_str(), "of" ) == 0 )
		{
			// <function> of <factor>
			pos++; // Increment our position past the "of"
			optr_inc++; // F***. God only knows...
			r = push_factor( pos, "~" );
			s = r.code;
			s += "<int ,const " + variant( i ).strVal() + ">";
		}
		else if( symbols[ pos ].mString == "(" )
		{
			// <function>( <expr1>, ... <exprN> )			
			s = "";
			s_tokens.push_back( "," );
			s_tokens.push_back( ")" );
			
			do
			{
				pos++; // Increment our position past the "("
				optr_inc++; // F***. God only knows...
				
				r = push_expr( pos, "~", s_tokens );
				s += r.code;
				pos = r.cur_ptr - 1;
				
			} while( r.found_token == "," );
			
			s += "<int ,const " + variant( i ).strVal() + ">";
			r.cur_ptr = pos + 1;
		}
		else
		{
			s += "<int ,const " + variant( i ).strVal() + ">";
			r.cur_ptr = pos;
		}
		
		if( oper != "~" )
		{
			s += "<merge ,#" + makeInlineOperator( oper ) + ">";
		}
		pos = r.cur_ptr - 1;
	}
	else if ( ! strcomp( symbols[ pos + 1 ].mString, "of" ) )
	{
		s += "<push const " + symbols[ pos ].mString + ",#nop>";
		s += ( r = push_factor ( pos + 2, "~" ) ).code;
		s += "<swap ,><int ,const 7003>";
		pos = r.cur_ptr - 1;
	}
	else
	{
	// yeah, yeah, yeah... I'll fix it later.
		// Ordinals...
		if ( strcmpi( symbols[ pos ].mString.c_str(), "zero" ) == 0 )
		{
			s += makeSimpleValue( oper, "0" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "one" ) == 0 )
		{
			s += makeSimpleValue( oper, "1" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "two" ) == 0 )
		{
			s += makeSimpleValue( oper, "2" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "three" ) == 0 )
		{
			s += makeSimpleValue( oper, "3" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "four" ) == 0 )
		{
			s += makeSimpleValue( oper, "4" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "five" ) == 0 )
		{
			s += makeSimpleValue( oper, "5" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "six" ) == 0 )
		{
			s += makeSimpleValue( oper, "6" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "seven" ) == 0 )
		{
			s += makeSimpleValue( oper, "7" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "eight" ) == 0 )
		{
			s += makeSimpleValue( oper, "8" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "nine" ) == 0 )
		{
			s += makeSimpleValue( oper, "9" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "ten" ) == 0 )
		{
			s += makeSimpleValue( oper, "10" );
		}
		else if ( ( strcmpi( symbols[ pos ].mString.c_str(), "quote" ) == 0 ) || ( strcmpi( symbols[ pos ].mString.c_str(), "quot" ) == 0 ) )
		{
			s += makeSimpleConstantValue( oper, "\"" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "return" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\r" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "linefeed" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\n" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "formfeed" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\12" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "empty" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\\n" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "space" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, " " );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "colon" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, ":" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "comma" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\\c" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "tab" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "\t" );
		}
		else if ( strcmpi( symbols[ pos ].mString.c_str(), "pi" ) == 0 )
		{
			s += makeSimpleConstantValue( oper, "3.14159265358979323846" );
		}
		else
		{
			if( symbols[ pos + 1 ].mString == "(" )
			{
				// Is function
				// <function>( <expr1>, ... <exprN> )			
				s = "";
				sx = "<fcall ,label f" + symbols[ pos ].mString + ">";
				
				if ( symbols[ pos + 2 ].mString != ")" )
				{
				
					s_tokens.push_back( "," );
					s_tokens.push_back( ")" );
					pos ++;
					do
					{
						pos++; // Increment our position past the "("
						optr_inc++; // F***. God only knows...
					
						// Parse the expression/factor/thingee.
						r = push_expr( pos, "~", s_tokens );
						s = r.code + s;
						pos = r.cur_ptr - 1;
					
					} while( r.found_token == "," );
				
				}
				else
				{
					pos += 3;
				}
				
				s = s + sx + "<push world .result,#nop>"; // Get it back on the stack.
				r.cur_ptr = pos + 1;
				if( oper != "~" )
				{
					s += "<merge ,#" + makeInlineOperator( oper ) + ">";
				}
			}
			else
			{
				// Is variable or constant.
				s += makeSimpleValue( oper, symbols[ pos ].mString );
			}
		}
	}
	
	// Reset the no_bracket_processing flag.
	no_bracket_processing = dontProcessTrailingBrackets;
	
	if ( ( symbols[ pos + 1 ].mString == "[" ) && ! no_bracket_processing )
	{
		// Bracket notation. Wheee!!!!
		pos += 2;
		s_tokens.clear();
		s_tokens.push_back ( "]" );
		
		// Push the string onto the stack.
		r = push_expr( pos, "~", s_tokens );
		s += r.code;
		pos = r.cur_ptr - 1;
		
		// Eat the string, and the object on the stack,
		// and spit out the value.
		s += "<int ,const 7003>";
	}	
	
	while( s2.size() > 0 )
	{
		if( s2.top() == "not" )
		{
			s += "<push const 0,#not>";
		}
		else if( s2.top() == "-" )
		{
			s += "<push const 0,#neg>";
		}
		else if( s2.top() == "+" )
		{
			// unary + does nothing;
		}
		s2.pop();
	}
	
	if( no_optimize )
	{
		s += "<merge ,#" + makeInlineOperator( backup_oper ) + ">";
	}
		
	return searchRecord( s, pos + 1, "", optr_inc );
}

short PXIHyperTalkCompiler::compare_precedence( const string &o1, const string &o2 )
{
    if ( ( *precedence.find( o1 ) ).second > ( *precedence.find( o2 ) ).second )
    {
        return 1;
    }
    else if ( ( *precedence.find( o1 ) ).second == ( *precedence.find( o2 ) ).second )
    {
		if( get_associativity( o1 ) && get_associativity( o2 ) )
		{
			return -1;
		}
		else
		{
			return 0;
		}
    }
    else
    {
        return -1;
    }
}

bool PXIHyperTalkCompiler::get_associativity( const string & oper )
{
	if(oper=="^")
	{
		return true;
	}
	else
	{
		return false;
	}
}

const string PXIHyperTalkCompiler::makeSimpleValue( const string &oper, const string &operand )
{	
	if ( is_a_number( operand ) || operand [ 0 ] == 1 )
	{
		// It's a keyword, so push it as a constant.
		return "<push const " + quotesrecord::c2escape( operand ) + ",#" + makeInlineOperator( oper ) + ">";
	}
	else
	{
		// It's obviously a variable.
		if ( haveCurrentGlobal ( capalllower( operand ) ) )
		{
			// It's a global variable.
			return "<push const " + variant( getCurrentGlobalIndex( capalllower( operand ) ) ).strVal() + ",#nop><int ,const 7000>" + ( ( makeInlineOperator( oper ) != "nop" ) ? ( "<merge ,#" + makeInlineOperator( oper ) + ">" ) : "" );
		}
		else
		{
			// It's a local variable.
			return "<push local " + operand + ",#" + makeInlineOperator( oper ) + ">";
		}
	}
}

const string PXIHyperTalkCompiler::makeSimpleContainerValue( const string &oper, const string &operand )
{
	if ( current_globals.find ( capalllower( operand ) ) != current_globals.end() )
	{
		// It's a global variable.
		if( makeInlineOperator( oper ) == "nop" )
		{
			// We can take the shortcut.
			// Put the name of the variable on the stack, and interrupt 7001 will eat
			// the name, and the next value on the stack, setting our variable for us.
			return "<push const " + capalllower( operand ) + ",#nop><int ,const 7001>";
		}
		else
		{
			// We don't have a simple operator.
			
			// Step 1: Pull the global out of storage:
			return "<push const " + variant( getCurrentGlobalIndex( capalllower( operand ) ) ).strVal() + ",#nop><int ,const 7000>" +
			
			// Step 2: Swap the top two values on the stack:
			string( "<swap ,>" ) +
			
			// Step 3: Merge them using the appropriate operator:
			"<merge ,#" + makeInlineOperator( oper ) + string( ">" ) +
			
			// Step 4: Put the variable name back on the stack
			// and call interrupt 7001:
			"<push const " + variant( getCurrentGlobalIndex( capalllower( operand ) ) ).strVal() + ",#nop><int ,const 7001>";
		}
	}
	else
	{
		// It's a local variable.
		return  "<check_variable ,local " + operand + "><pop local " + operand + ",#" + makeInlineOperator( oper ) + ">";
	}
}

const string PXIHyperTalkCompiler::makeSimpleConstantValue( const string &oper, const string &operand )
{
	// Creates a reference, but forces a constant. Useful for situations where constants
	// can look like variables.
    return string( "<push const " ) + operand + ",#" + makeInlineOperator( oper ) + ">";
}

const string PXIHyperTalkCompiler::makeInlineOperator( const string &oper )
{
    if ( strcmp( oper.c_str(), "+" ) == 0 )
    {
        return "add";
    }
    else if ( !strcmpi( oper.c_str(), "-" ) )
    {
        return "sub";
    }
    else if ( !strcmpi( oper.c_str(), "*" ) )
    {
        return "mul";
    }
    else if ( !strcmpi( oper.c_str(), "/" ) )
    {
        return "div";
    }
	else if ( !strcmpi( oper.c_str(), "^" ) )
	{
		return "pow";
	}
    else if ( !strcmpi( oper.c_str(), "=" ) )
    {
        return "cmp";
    }
    else if ( !strcmpi( oper.c_str(), ">" ) )
    {
        return "gtx";
    }
	else if ( !strcmpi( oper.c_str(), ">=" ) )
	{
		return "gte";
	}
    else if ( !strcmpi( oper.c_str(), "<" ) )
    {
        return "ltx";
    }
    else if ( !strcmpi( oper.c_str(), "<=" ) )
    {
        return "lte";
    }
    else if ( !strcmpi( oper.c_str(), "is" ) )
    {
        return "cmp";
    }
	else if ( !strcmpi( oper.c_str(), "is not" ) )
	{
		return "neq";
	}
	else if ( !strcmpi( oper.c_str(), "isn't" ) )
	{
		return "neq";
	}
	else if( !strcmpi( oper.c_str(), "contains" ) )
	{
		return "has";
	}
	else if( !strcmpi( oper.c_str(), "is in" ) )
	{
		return "hs2";
	}
	else if( !strcmpi( oper.c_str(), "does not contain" ) )
	{
		return "hs3";
	}
	else if( !strcmpi( oper.c_str(), "is not in" ) )
	{
		return "hs4";
	}
    else if ( !strcmpi( oper.c_str(), "~" ) )
    {
        return "nop";
    }
    else if ( !strcmpi( oper.c_str(), "&" ) )
    {
        return "cat";
    }
    else if ( !strcmpi( oper.c_str(), "&&" ) )
    {
        return "cct";
    }
    else if ( !strcmpi( oper.c_str(), "and" ) )
    {
        return "and";
    }
    else if ( !strcmpi( oper.c_str(), "mod" ) )
    {
        return "mod";
    }
    else if ( !strcmpi( oper.c_str(), "div" ) )
    {
        return "div";
    }
	else if ( !strcmpi( oper.c_str(), "or" ) )
	{
		return "or ";
	}
    else
    {
        return "nop";
    }
}

bool PXIHyperTalkCompiler::is_a_number( const string &s )
{
    int i;
    for ( i = 0;i < ( int ) s.size();i++ )
    {
        if ( ( !isdigit( s[ i ] ) ) && ( s[ i ] != '.' ) )
        {
            return false;
        }
    }
    return true;
}

bool PXIHyperTalkCompiler::is_an_operator( const string &s )
{
	if( strcmpi( s.c_str(), "+" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "-" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "*" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "/" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "~" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "^" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), ">" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), ">=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "&" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<>" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is not" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "isn't" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "contains" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is in" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "does not contain" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is not in" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "&&" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "and" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "mod" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "div" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "or" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "(" ) == 0 )
	{
		return true;
	}
	else if( is_a_chunk_word( s ) )
	{
		return true;
	}
	return false;
}

bool PXIHyperTalkCompiler::is_not_a_factor( const string &s )
{
	if( strcmpi( s.c_str(), "+" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "-" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "*" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "/" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "~" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "^" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), ">" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), ">=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "&" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "=" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "<>" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is not" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "isn't" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "contains" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is in" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "does not contain" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "is not in" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "&&" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "and" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "mod" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "div" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "or" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "(" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), ")" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "@" ) == 0 )
	{
		return true;
	}
	return false;
}

bool PXIHyperTalkCompiler::is_a_chunk_word( const string &s )
{
	if( strcmpi( s.c_str(), "char" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "character" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "word" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "line" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "item" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "sent" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "sentence" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "para" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "paragraph" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "elem" ) == 0 )
	{
		return true;
	}
	else if( strcmpi( s.c_str(), "element" ) == 0 )
	{
		return true;
	}
	else
	{
		return false;
	}
}

int PXIHyperTalkCompiler::is_a_shortfunc( const string & s )
{
	if( strcmpi( s.c_str(), "chartonum" ) == 0 )
	{
		return 4005;
	}
	else if( strcmpi( s.c_str(), "numtochar" ) == 0 )
	{
		return 4043;
	}
	else if( strcmpi( s.c_str(), "length" ) == 0 )
	{
		return 4025;
	}
	else if( strcmpi( s.c_str(), "ticks" ) == 0 )
	{
		return 4068;
	}
	else if( strcmpi( s.c_str(), "seconds" ) == 0 )
	{
		return 4052;
	}
	else if( strcmpi( s.c_str(), "sum" ) == 0 )
	{
		return 5000;
	}
	else if( strcmpi( s.c_str(), "userlevel" ) == 0 )
	{
		return 5001;
	}
	else if( strcmpi( s.c_str(), "random" ) == 0 )
	{
		return 4048;
	}
	else if( strcmpi( s.c_str(), "result" ) == 0 )
	{
		return 4049;
	}
	else
	{
		return 0;
	}
}

bool PXIHyperTalkCompiler::is_tens_place_ordinal( const string & s )
{
	if ( instr( " twenty thirty forty fifty sixty seventy eighty ninety ", " " + s + " " ) >= 0 )
	{
		return true;
	}
	return false;
}

bool PXIHyperTalkCompiler::is_ones_place_ordinal( const string & s )
{
	if ( instr( " first second third fourth fifth sixth seventh eigth ninth ", " " + s + " " ) >= 0 )
	{
		return true;
	}
	return false;
}

int PXIHyperTalkCompiler::shortfunc( const string & s )
{
	return is_a_shortfunc( s );
}

void PXIHyperTalkCompiler::echo( void )
{
    cout << composite_asmcode << endl;
}

void PXIHyperTalkCompiler::echo_tokens( void )
{
    vector < Token > ::iterator i;
    for ( i = symbols.begin();i != symbols.end();++i )
    {
        cout << ( *i ).mString << endl;
    }
}

bool PXIHyperTalkCompiler::is_a_padding_symb( const char c )
{
    switch ( c )
    {
    case ' ':
        return true;
        break;
    case '(':
        return true;
        break;
    case ')':
        return true;
        break;
    case '+':
        return true;
        break;
    case '-':
        return true;
        break;
    case '*':
        return true;
        break;
    case '/':
        return true;
        break;
    case '&':
        return true;
        break;
    case '^':
        return true;
        break;
    case '>':
        return true;
        break;
    case '<':
        return true;
        break;
    case '=':
        return true;
        break;
    default:
        return false;
        break;
    }
}

void PXIHyperTalkCompiler::write_symbols( void )
{

	no_bracket_processing = false;
	
	precedence.insert( pair < string, int > ( "not", 7 ) );
	precedence.insert( pair < string, int > ( "^", 7 ) );
    precedence.insert( pair < string, int > ( "*", 6 ) );
    precedence.insert( pair < string, int > ( "/", 6 ) );
    precedence.insert( pair < string, int > ( "div", 6 ) );
    precedence.insert( pair < string, int > ( "mod", 6 ) );
    precedence.insert( pair < string, int > ( "+", 5 ) );
    precedence.insert( pair < string, int > ( "-", 5 ) );
    precedence.insert( pair < string, int > ( "<", 4 ) );
    precedence.insert( pair < string, int > ( ">", 4 ) );
    precedence.insert( pair < string, int > ( "<=", 4 ) );
    precedence.insert( pair < string, int > ( ">=", 4 ) );
    precedence.insert( pair < string, int > ( "=", 3 ) );
    precedence.insert( pair < string, int > ( "is", 3 ) );
    precedence.insert( pair < string, int > ( "<>", 3 ) );
    precedence.insert( pair < string, int > ( "is not", 3 ) );
    precedence.insert( pair < string, int > ( "isn't", 3 ) );
    precedence.insert( pair < string, int > ( "&", 2 ) );
    precedence.insert( pair < string, int > ( "&&", 2 ) );
    precedence.insert( pair < string, int > ( "is in", 1 ) );
    precedence.insert( pair < string, int > ( "is not in", 1 ) );
    precedence.insert( pair < string, int > ( "does not contain", 1 ) );
    precedence.insert( pair < string, int > ( "contains", 1 ) );
    precedence.insert( pair < string, int > ( "and", 0 ) );
    precedence.insert( pair < string, int > ( "or", 0 ) );
    precedence.insert( pair < string, int > ( "~", 0 ) );
    precedence.insert( pair < string, int > ( "@", -1 ) );
	precedence.insert( pair < string, int > ( "$", -1 ) );
	precedence.insert( pair < string, int > ( "of", 99 ) );
	precedence.insert( pair < string, int > ( ")", -1 ) );

	incrementor_symbs.push_back( "char" );
	incrementor_symbs.push_back( "character" );
	incrementor_symbs.push_back( "word" );
	incrementor_symbs.push_back( "line" );
	incrementor_symbs.push_back( "item" );
	incrementor_symbs.push_back( "para" );
	incrementor_symbs.push_back( "paragraph" );
	incrementor_symbs.push_back( "byte" );
	incrementor_symbs.push_back( "short" );
	incrementor_symbs.push_back( "long" );
	incrementor_symbs.push_back( "sent" );
	incrementor_symbs.push_back( "sentence" );
	incrementor_symbs.push_back( "list" );
	incrementor_symbs.push_back( "elem" );
	incrementor_symbs.push_back( "elements" );
	incrementor_symbs.push_back( "number" );
	incrementor_symbs.push_back( "(" );
	
	decrementor_symbs.push_back( "in" );
	decrementor_symbs.push_back( "of" );
	decrementor_symbs.push_back( ")" );
	
	continuable_symbs.push_back( "the" );
	continuable_symbs.push_back( "first" );
	continuable_symbs.push_back( "last" );
	continuable_symbs.push_back( "middle" );
	continuable_symbs.push_back( "mid" );
	continuable_symbs.push_back( "mdd" );
	continuable_symbs.push_back( "any" );
	continuable_symbs.push_back( "some" );
}

string PXIHyperTalkCompiler::rtrim( const string &s )
{
    string n = s;
    return n.erase( n.find_last_not_of( " " ) + 1 );
}

string PXIHyperTalkCompiler::ltrim( const string &s )
{
    string n = s;
    return n.erase( 0, s.find_first_not_of( " " ) );
}

string PXIHyperTalkCompiler::trim( const string &s )
{
    string n = s;
    return ltrim( rtrim( n ) );
}

int PXIHyperTalkCompiler::strcmpi( const char* s1, const char* s2 )
{
    char c1, c2;
    while ( 1 )
    {
        c1 = tolower( *s1++ );
        c2 = tolower( *s2++ );
        if ( c1 < c2 )
            return -1;
        if ( c1 > c2 )
            return 1;
        if ( c1 == 0 )
            return 0;
    }
}

int PXIHyperTalkCompiler::stricmp( const char* str1, const char* str2 )
{
    return strcmpi( str1, str2 );
}

const string & PXIHyperTalkCompiler::get_source( void )
{
	return source;
}

bool PXIHyperTalkCompiler::haveCurrentGlobal( const string & key )
{
	return current_globals.find( key ) != current_globals.end();
}

int PXIHyperTalkCompiler::getCurrentGlobalIndex( const string & key )
{
	return ( * current_globals.find( key ) ).second;
}